﻿using Microsoft.Xrm.Sdk;
using Microsoft.Xrm.Sdk.Messages;
using Microsoft.Xrm.Sdk.Query;
using System;
using System.ServiceModel;
using VA.PPMS.CRM.Plugins.Helper;
using VA.PPMS.CRM.Plugins.SamExclusionSearch;

namespace VA.PPMS.CRM.Plugins
{
    public class ProviderIdentifierCreate : IPlugin
    {
        private const string ProcessDateAttribute = "ppms_processdateleie";
        private const string PluginName = "ProviderIdentifierCreate";
        private IOrganizationService _service;
        private ITracingService _tracingService;

        /// <summary>
        /// Plug-in entry point, called when criteria are met
        /// </summary>
        /// <param name="serviceProvider"></param>
        public void Execute(IServiceProvider serviceProvider)
        {
            // Tracing service for debugging
            _tracingService = (ITracingService)serviceProvider.GetService(typeof(ITracingService));

            // Get execution context
            IPluginExecutionContext context = (IPluginExecutionContext)serviceProvider.GetService(typeof(IPluginExecutionContext));

            if (context.InputParameters.Contains("Target") && context.InputParameters["Target"] is Entity)
            {
                _tracingService.Trace("Begin");

                // Obtain the target entity from the input parameters.
                Entity target = (Entity)context.InputParameters["Target"];

                // Verify target entity type
                if (target.LogicalName != "ppms_provideridentifier")
                    return;

                _tracingService.Trace("Entity found: {0}", target.Id.ToString());

                // Get organization service reference
                IOrganizationServiceFactory serviceFactory = (IOrganizationServiceFactory)serviceProvider.GetService(typeof(IOrganizationServiceFactory));
                _service = serviceFactory.CreateOrganizationService(context.UserId);

                try
                {
                    // Check for NPI type
                    OptionSetValue providerIdType = target.GetAttributeValue<OptionSetValue>("ppms_provideridentifiertype");


                    switch (providerIdType.Value)
                    {
                        case (int)PpmsHelper.ProviderIdentifierType.NPI:
                            ValidateNpi(target);
                            break;
                        case (int)PpmsHelper.ProviderIdentifierType.TIN:
                            ValidateTin(target);
                            break;
                        default:
                            _tracingService.Trace("Target entity is not an NPI or TIN.");
                            break;
                    }
                }
                catch (FaultException<OrganizationServiceFault> ex)
                {
                    _tracingService.Trace("Fault: {0}", ex.ToString());
                    throw new InvalidPluginExecutionException(String.Format("An error occurred in {0}.", PluginName), ex);
                }
                catch (Exception ex)
                {
                    _tracingService.Trace("Exception: {0}", ex.ToString());
                    throw;
                }
            }
            _tracingService.Trace("Done");
        }

        /// <summary>
        /// Validate the NPI value LEIE and NPPES lists
        /// </summary>
        /// <param name="target">ProviderIdentifier entity</param>
        private void ValidateNpi(Entity target)
        {
            OptionSetValue providerNpiType = target.GetAttributeValue<OptionSetValue>("ppms_npitype");
            if (providerNpiType == null)
            {
                _tracingService.Trace("Target entity does not contain Npi Type.");
                return;
            }

            string npiType;
            switch (providerNpiType.Value)
            {
                case (int)PpmsHelper.ProviderNpiType.Individual:
                    npiType = "1";
                    break;
                case (int)PpmsHelper.ProviderNpiType.GroupAgency:
                    npiType = "2";
                    break;
                case (int)PpmsHelper.ProviderNpiType.Organizational:
                    npiType = "2";
                    break;
                default:
                    _tracingService.Trace("Npi Type not found.");
                    return;
            }

            // Get npi with provider reference
            var entity = GetNpi(target);
            if (entity == null)
            {
                _tracingService.Trace("Target entity not found.");
                return;
            }

            // Get associated provider reference
            var provider = entity.GetAttributeValue<EntityReference>("ppms_provider");
            // Get current batch detail
            var detail = GetCurrentBatchDetail(provider);
            var npi = entity.GetAttributeValue<string>("ppms_provideridentifier");

            // Lookup validation setting
            _tracingService.Trace("Retrieve entity");
            EntityCollection result = GetMatchingExclusions(npi);
            if (result != null && result.Entities.Count > 0 && detail != null)
            {
                _tracingService.Trace("Execute deactivate request");
                _service.Execute(PpmsHelper.GetDeactivateRequest(provider, (int)PpmsHelper.Account_StatusCode.LeieExclusion));
                // Create batch detail result for validation failure
                _tracingService.Trace("Create batch detail result");
                CreateBatchDetailLeieResult(detail.Id);
            }

            // Lookup NPPES status
            _tracingService.Trace("Check NPI Status: Npi{0} , NpiType{1}", npi, npiType);

            bool isActive = CheckNpiStatus(npi, npiType);
            if (!isActive)
            {
                _tracingService.Trace("Execute deactivate request");
                _service.Execute(PpmsHelper.GetDeactivateRequest(provider, (int)PpmsHelper.Account_StatusCode.NpiCheckFailure));

                // Create batch detail result for validation failure
                if (detail != null)
                {
                    _tracingService.Trace("NPI - Create batch detail result");
                    CreateBatchDetailNpiCheckResult(detail.Id);
                }
            }

            // Create NPI validation entry
            _tracingService.Trace("Create NPPES validation entries...");
            PpmsHelper.AddProviderValidation(_service, provider, PpmsHelper.Validation_Type.Nppes);
        }


        private void ValidateTin(Entity target)
        {
            // Get npi with provider reference
            var entity = GetNpi(target);
            if (entity == null)
            {
                _tracingService.Trace("Target entity not found.");
                return;
            }

            // Get associated provider reference
            _tracingService.Trace("Retrieve entity");
            var provider = entity.GetAttributeValue<EntityReference>("ppms_provider");
            // Get current batch detail
            var detail = GetCurrentBatchDetail(provider);
            var tin = entity.GetAttributeValue<string>("ppms_provideridentifier");

            // Lookup SAM.gov status
            _tracingService.Trace("Check SAM Status: TIN{0}", tin);

            bool isExcluded = GetExclusionStatus(tin, provider.Name);
            if (!isExcluded)
            {
                _tracingService.Trace("Execute deactivate request");
                _service.Execute(PpmsHelper.GetDeactivateRequest(provider, (int)PpmsHelper.Account_StatusCode.SamsExclusion));

                // Create batch detail result for validation failure
                if (detail != null)
                {
                    _tracingService.Trace("SAM - Create batch detail result");
                    CreateBatchDetailSamCheckResult(detail.Id);
                }
            }

            // Create SAM validation entry
            _tracingService.Trace("Create SAM validation entries...");
            PpmsHelper.AddProviderValidation(_service, provider, PpmsHelper.Validation_Type.Sam);
        }

        private Entity GetNpi(Entity entity)
        {
            if (entity != null)
            {
                return _service.Retrieve("ppms_provideridentifier", entity.Id, new ColumnSet(new string[] { "ppms_provideridentifier", "ppms_provider" }));
            }

            return null;
        }

        private EntityCollection GetMatchingExclusions(string npi)
        {
            FilterExpression filter = new FilterExpression();
            filter.AddCondition("ppms_npi", ConditionOperator.Equal, npi);
            filter.AddCondition("ppms_action", ConditionOperator.Equal, (int)PpmsHelper.LeieAction.Exclude);
            filter.AddCondition("statecode", ConditionOperator.Equal, (int)PpmsHelper.AccountState.Active);

            QueryExpression query = new QueryExpression("ppms_leieexclusion");
            query.ColumnSet.AddColumns("ppms_npi", "ppms_action");
            query.Criteria.AddFilter(filter);

            return _service.RetrieveMultiple(query);
        }

        private void CreateBatchDetailLeieResult(Guid detailId)
        {
            // Add detail to batch
            var result = new Entity("ppms_batchdetailresult");
            result.Attributes.Add("ppms_name", "LEIE Exclusion");
            result.Attributes.Add("ppms_isvalid", false);
            result.Attributes.Add("ppms_entitytype", "Provider");
            result.Attributes.Add("ppms_result", "Affected by LEIE Exclusion rules");
            result.Attributes.Add("ppms_message", "The provider is on the current LEIE exclusion list.");
            result.Attributes.Add("ppms_batchdetail", new EntityReference("ppms_batchdetail", detailId));
            _service.Create(result);
        }

        private void CreateBatchDetailNpiCheckResult(Guid detailId)
        {
            // Add detail to batch
            var result = new Entity("ppms_batchdetailresult");
            result.Attributes.Add("ppms_name", "NPI Status Check");
            result.Attributes.Add("ppms_isvalid", false);
            result.Attributes.Add("ppms_entitytype", "Provider");
            result.Attributes.Add("ppms_result", "Affected by NPI status check rules");
            result.Attributes.Add("ppms_message", "The provider is not on the active list.");
            result.Attributes.Add("ppms_batchdetail", new EntityReference("ppms_batchdetail", detailId));
            _service.Create(result);
        }

        private void CreateBatchDetailSamCheckResult(Guid detailId)
        {
            // Add detail to batch
            var result = new Entity("ppms_batchdetailresult");
            result.Attributes.Add("ppms_name", "SAM Status Check");
            result.Attributes.Add("ppms_isvalid", false);
            result.Attributes.Add("ppms_entitytype", "Provider");
            result.Attributes.Add("ppms_result", "Affected by SAM status check rules");
            result.Attributes.Add("ppms_message", "The provider is not on the active list.");
            result.Attributes.Add("ppms_batchdetail", new EntityReference("ppms_batchdetail", detailId));
            _service.Create(result);
        }

        private Entity GetBatchDetail(Guid batchDetailId)
        {
            return _service.Retrieve("ppms_batchdetail", batchDetailId, new ColumnSet(new string[] { "ppms_batchdetailid", "ppms_name", "ppms_processdateleie", "ppms_batch" }));
        }

        private Entity GetProviderWithBatchDetail(Guid providerId)
        {
            var request = new RetrieveRequest();

            request.Target = new EntityReference("account", providerId);
            request.ColumnSet = new ColumnSet(new string[] { "accountid", "name", "statecode" });
            request.RelatedEntitiesQuery = new RelationshipQueryCollection();

            //Retrieve related entities
            request.RelatedEntitiesQuery.Add(new Relationship("ppms_account_batchdetail_provider"),
                new QueryExpression("ppms_batchdetail")
                {
                    ColumnSet = new ColumnSet("ppms_batchdetailid", "ppms_name", "ppms_batch", "ppms_processdateleie")
                }
            );

            //Get response
            var response = (RetrieveResponse)_service.Execute(request);
            if (response != null)
                return response.Entity;

            return null;
        }

        private Entity GetCurrentBatchDetail(EntityReference providerRef)
        {
            if (providerRef != null)
            {
                var provider = GetProviderWithBatchDetail(providerRef.Id);
                if (provider != null)
                {
                    // determine target batch detail
                    var children = provider.RelatedEntities[new Relationship("ppms_account_batchdetail_provider")];
                    if (children != null && children.Entities != null)
                    {
                        // find active batch detail 
                        foreach (var child in children.Entities)
                        {
                            var processDate = child.GetAttributeValue<DateTime?>(ProcessDateAttribute);
                            if (!processDate.HasValue)
                            {
                                // retrieve entity and update process date
                                return GetBatchDetail(child.Id);
                            }
                        }
                    }
                }
            }

            return null;
        }

        private bool CheckNpiStatus(string npi, string npiType)
        {
            //The streamreader always returns a string even if there is no matching NPI 
            //so will do a check if Npi value is present. 
            var npiData = FabricHelper.GetNpiStatus(npi, npiType, _tracingService);

            _tracingService.Trace("Checking for Npi value {0} in NpiData: {1}", npi, npiData);
            if (npiData.Contains(npi))
            {
                _tracingService.Trace("Macthing Npi value found: {0}", npiData);
                return true;
            }
            _tracingService.Trace("No Matching Npi value found {0}", npiData);
            return false;
        }

        /// <summary>
        /// Returns the exlcusion status for the provider based on SAM.gov information
        /// </summary>
        /// <param name="tin">Tax Identification Number</param>
        /// <param name="providerName">Name of the provider</param>
        /// <returns>true if found on the exclusion list, or if error occurs retrieving information</returns>
        private bool GetExclusionStatus(string tin, string providerName)
        {
            try
            {
                var op = new OperationExSSNSearchType();
                op.exactName = providerName;
                op.ssnOrTin = tin;

                // Search for exclusions
                var wf = new ExclusionSearchServiceBeanService();
                var response = wf.doSsnSearch(op);
                if (response.transactionInformation.successful)
                {
                    if (response.excludedEntity != null)
                    {
                        // Note exclusion types reported
                        foreach (var item in response.excludedEntity)
                        {
                            _tracingService.Trace($"Result: {item.exclusionType}");
                        }
                    }
                    else
                    {
                        // No exclusions returned
                        _tracingService.Trace("Exclusion not found.");
                        return true;
                    }
                }
                else
                {
                    _tracingService.Trace("Unable to access Exclusion Search Service.");
                }

                return false;
            }
            catch (Exception ex)
            {
                _tracingService.Trace($"ERROR: {ex.Message}");
            }

            return false;
        }
    }
}
